<?php
/*
	Simple PHP Debug Class
.---------------------------------------------------------------------------.
|  Software: Debug - Simple PHP Debug Class                                 |
|   Version: 1.95                                                           |
|      Site: http://jspit.de/?page=debug                                    |
| ------------------------------------------------------------------------- |
| Copyright © 2010-2013, Peter Junk (alias jspit). All Rights Reserved.     |
| ------------------------------------------------------------------------- |
|   License: Distributed under the Lesser General Public License (LGPL)     |
|            http://www.gnu.org/copyleft/lesser.html                        |
| This program is distributed in the hope that it will be useful - WITHOUT  |
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     |
| FITNESS FOR A PARTICULAR PURPOSE.                                         |
'---------------------------------------------------------------------------'
  Last modify : 2016-01-24
  2013-02-25: add function strhex 
  2013-05-29: new stop-Method
  2013-06-19: +DOM
  2013-07-03: + detect_encoding
  2013-08-08: mit debug::$real_time_output = true wird das output_buffering überlistet
  2013-09-17: isset closure bug -> property_exists
  2013-09-23: more file infos
  2013-09-24: + exitOnStop Attribut (V1.5)
  2013-12-17: detect_encoding +
  2014-01-22: + crc checksum
  2014-02-06: + fkt,classfkt 
  2014-02-20: + Windows-1252 detect encoding
  2014-03-25: + setTitleColor, wrc (write with bg-color = red)
  2014-07-10: stdClass::__set_state -> (object)
  2014-10-31: + getClean()
  2015-02-05: + debug::$stringCut  (80 default)
  2015-12-11: modify str2print + detect_encoding
*/
if (version_compare(PHP_VERSION, '5.3.0', '<') ) exit("Simple PHP Debug Class requires at least PHP version 5.3.0!\n");
if (isset($_SERVER["SCRIPT_NAME"]) && basename($_SERVER["SCRIPT_NAME"])==basename(__FILE__))die();

class Debug
{
  /*
   
  From the Simple Debug Class you get timestamp, backtrace-info and var-info's in table view
  on display or in a logfile. The methods write, save and stop accepts a variable number of parameters.
  It is very easy to use. Some examples:
  
  * Example 1 : Show Debug-Info from scalar/array/object.. on screen  
    Debug::write($var1,$var2); 
    
  * Example 2 : Write Debug-Info into logfile 
    Debug::log("logfilename");  //Script errors are also logged in the file if the second parameter is true
    Debug::write($var1,$var2); 
    Debug::log("");  //Close logfile, next write will show on screen

  * Example 3 : Disable all functions (Debug/Log-off)
    Debug::log(false);
    Debug::log(true);   //Debug/Log on

  * Example 4 : Save Write Debug-Info into internal Buffer (User-Output is not touched) 
    echo "useroutput";
    Debug::save($var1,$var2); //no output
    echo "useroutput";
    Debug::write($var3,$var4); //show saved and current info or save in logfile

  * Example 4 : Clear the internal buffer, if it is not longer needed
    Debug:clear();

  * Example 4 : Stop (Write and exit);
    Debug:stop($condition,$var1,$var2);

  */
  
  const VERSION = "1.95";
  //convert special chars in hex-code
  public static $showSpecialChars = true;           
  //Liefert bei aktivierter Ausgabepufferung die debug-informationen zeitnah auf dem Display
  public static $real_time_output = false;
  //Ist $exitOnStop true, wird bei der Stop-Methode nach der Ausgabe ein exit vollzogen
  //Wird $exitOnStop auf false gesetzt, erfolgt bei der Stop-Methode nach der Ausgabe ein return true
  public static $exitOnStop = true;
  //cut Strings with more than $stringCut chars and $showSpecialChars = true
  public static $stringCut = 180;

  protected static $tableStyle = 'border:1px solid #3C3733;border-collapse:collapse;font:normal 12px Arial; width:98%;text-align:left;margin:2px;'; 
  protected static $trHeadStyle = 'border:1px solid #3C3733;background-color:#36f;color:#fff';  
  protected static $tdStyle = 'border:1px solid #3C3733;vertical-align:top;background:#fff;color:#000;';
  protected static $col1width = '30px';
  protected static $col2width = '165px';
  //
  protected static $debug_on_off = true;
  protected static $logfilename = "";
  protected static $recbuf = "";
  protected static $lastTime = 0;
  //system error log in the same file
  protected static $sys_err_log = false;
  protected static $log_errors = "";
  protected static $error_log = "";
  protected static $log_file_append = false;
  //
  protected static $stopCounter = null;

  
  /*
  * start/stop logging display/logfile
  * log(false)          : all methods will do nothing , System-Log unchanged
  * log("filename")     : start and write debug-info into logfile
  * log("+filename")    : start and write debug-info into logfile append
  * log("filename",true): start and write debug-info and system Warnings and Errors into logfile
  * log('')             : stop logfile, next write will show on display
  */
  public static function log($OnOff_or_File = true, $OnOff_err_log = null) 
  {
    if(is_bool($OnOff_or_File)) self::$debug_on_off = $OnOff_or_File;  //Argument is true/false
    elseif(is_string($OnOff_or_File)) 
    { //LogFile
      if($OnOff_or_File != "") 
      { // logging part
        $filename = $OnOff_or_File;
        if( self::$log_file_append = (substr($filename,0,1)=== '+')) $filename = substr($filename,1);
        if( !self::$log_file_append || !file_exists($filename)) {
          //create a new logfile
          $content = '<!DOCTYPE html>'."\r\n".
            '<html><head>'."\r\n".
            '<meta http-equiv="content-type" content="text/html;charset=utf-8">'."\r\n".
            '<title>Debug Log '.$OnOff_or_File.'</title>'."\r\n".
            '</head><body>'."\r\n";
          file_put_contents($filename,$content);  
          if((fileperms($filename) & 0666) != 0666) chmod($filename,0666); //+rw all
        }
        self::$logfilename = $filename;
      }
      else 
      { //empty string causes stop logging 
        self::closeLog();
      }
    }
    //check sys err log set off
    if(self::$sys_err_log && ($OnOff_err_log === false || $OnOff_or_File === '')) {
      //set back
      ini_set('log_errors', self::$log_errors);
      ini_set('error_log', self::$error_log);
    }
    //OnOff_err_log
    if(is_bool($OnOff_err_log)) {
      if($OnOff_err_log && self::$logfilename != "") {
          self::$log_errors = ini_get('log_errors');
          ini_set('log_errors', 'On');
          self::$error_log = ini_get('error_log');
          ini_set('error_log', self::$logfilename);
        }
      self::$sys_err_log = $OnOff_err_log;
    }
    //
    if(self::$lastTime == 0) self::$lastTime = microtime(true);  //save timestamp
  }
  
  /*
  * save the loginformation in a buffer or logfile, no output
  */
  public static function save(/** $var1, $var2, .. **/) 
  {
    if (!self::$debug_on_off) return;
    $argv = func_get_args();
    $backtrace = debug_backtrace();
    self::$recbuf .= self::recArg($argv,$backtrace);
  }

  /*
  * general output for saved and current debug-infos on display or logfile
  */
  public static function write(/** $var1, $var2, .. **/) 
  {
    if (!self::$debug_on_off) return;  //do nothing if $debug_on_off == false
    $argv = func_get_args();
    $backtrace = debug_backtrace();
    self::displayAndLog($argv,$backtrace);  
  }
  
  //write with color red
  public static function wrc(/** $var1, $var2, .. **/) 
  {
    if (!self::$debug_on_off) return;  //do nothing if $debug_on_off == false
    $defaultFormat = self::$trHeadStyle;
    self::setTitleColor('#a00');
    $argv = func_get_args();
    $backtrace = debug_backtrace();
    self::displayAndLog($argv,$backtrace); 
    self::$trHeadStyle = $defaultFormat;    
  }
  
  //get the loginformation from buffer and delete buffer
  //return html-table
  public static function getClean(/** $var1, $var2, .. **/) 
  {
    $argv = func_get_args();
    $backtrace = debug_backtrace();
    self::$recbuf .= self::recArg($argv,$backtrace);  //save current info
    $content = self::$recbuf;
    self::$recbuf = "";
    return $content;
  }


  /*
  * general output like write a output for saved and actual debug-infos on display or logfile and exit
  * $condition : if condition ist true or null then write and exit, in the other case do nothing
  * if $condition is int, write and stop at the $condition call
  * if $condition is int and 0 or <0 do nothing (also no  decrement stopCounter)
  */
  public static function stop($condition = null/**, $var1, $var2, .. **/) 
  {
    if (!self::$debug_on_off) return null;  //do nothing if $debug_on_off == false
    $argv = func_get_args();
    if($condition !== null) {
      array_shift($argv);
      if(is_bool($condition)) {
        if(!$condition) return false; //$condition is false
      }
      else {
        $maxZyk = (int) $condition;
        if($maxZyk <= 0) return false;  //0 und negative Zahlen haben keine Wirkung
        if(self::$stopCounter === null) self::$stopCounter = $maxZyk -1;
        --self::$stopCounter;
        if(self::$stopCounter >= 0) return false;
      }
    }
    $backtrace = debug_backtrace();
    self::displayAndLog($argv,$backtrace); 
    self::closeLog();
    if(self::$exitOnStop) {
      exit();
    }
    return true;
  }

  /*
  * Reset for Stop Counter
  */
  public static function resetStopCounter() 
  {
    self::$stopCounter = null;
  }
  
  
  /*
  * clear internal buffer, logfile-infos not delete
  */
  public static function clear() 
  {
    self::$recbuf = "";
  }
  
  //Info
  public static function systeminfo() {
    $info = PHP_OS.' PHP '.PHP_VERSION.' ('.(PHP_INT_SIZE * 8).'Bit) ';
    self::write($info);
  }
  
  /*
   * Liefert die 32Bit CRC Prüfsumme vom Argument als Hex-String mit der Länge 8
   * das Argument kann jeder Datentyp (string, array, Objekt) außer Resource sein
   * 
   */
  public static function crc($value) {
    return sprintf('%X',crc32(json_encode($value)));
  }
  
  /*
  * TypeInfo returns a string type of (len,class,resource-typ..)
  */
  public static function TypeInfo($obj) {
    $objType = gettype($obj);
    if(is_string($obj)) {
      $encoding = (string) self::detect_encoding($obj);
      return $objType."(".strlen($obj).") ".$encoding;
    }
    if(is_array($obj)) return  $objType."(".count($obj).")";
    if(is_object($obj)) {
      $objlenght = max(count($obj),count((array)$obj),(property_exists($obj,'length') ? $obj->length : 0));
      return $objType."(".get_class($obj).")(".$objlenght.")"; //2013
      }
    if(is_resource($obj)) return $objType."(".get_resource_type($obj).")"; 
    return $objType;
  }
   
  
  /*
   * Liefert string als '\x61\x62..'
   */
  public static function strhex($s){
    return $s != '' ? '\\x'.implode('\\x',str_split(bin2hex($s),2)) : '';
  }

  /*
   * Liefert für Unicode im Format U+20ac das Zeichen
   */
  public static function UniDecode($strUplus){
    $strUplus = preg_replace("/U\+([0-9A-F]{4})/i", "&#x\\1;", $strUplus); //4 Hexzahlen nach U+ rausfiltern
    return html_entity_decode($strUplus, ENT_NOQUOTES, 'UTF-8');
   }
  
  /*
   * liefert 'UTF-8','UTF-8 BOM','ISO-8859-1','ASCII' oder false
   * faster as mb_detect_encoding($string,'ASCII, UTF-8, ISO-8859-1',true);
   */
  public static function detect_encoding($string) {
    if( preg_match("/^\xEF\xBB\xBF/",$string)) return 'UTF-8 BOM';
    if( preg_match("//u", $string)) {
      if( preg_match("/[\xf0-\xf7][\x80-\xbf][\x80-\xbf][\x80-\xbf]/",$string)) return 'UTF-8mb4';
      if( preg_match("/[\xe0-\xef][\x80-\xbf][\x80-\xbf]/",$string)) return 'UTF-8mb3';
      if( preg_match("/[\x80-\xff]/",$string)) return 'UTF-8';
      return 'ASCII';
    } else {
      if(preg_match("/[\x7f-\x9f]/",$string)) {
        //kein ISO/IEC 8859-1
        if(preg_match("/[\x81\x8d\x90\x9d]/",$string)) {
          //kein CP1252
          return "";
        }
        return 'CP1252';
      }
      return 'ISO-8859-1';
    }
    return false;
  }
  
  /*
   * check callbackfkt
   */
  public static function fkt($fctName/** $var1, $var2, .. **/){
    //time empty function
    $fctDummy = function(){};
    $microtime_start = microtime(true);
    $r = call_user_func_array($fctDummy,array());
    $diff_microsec_dummy = microtime(true)-$microtime_start;
    
    $argv = func_get_args();
    array_shift($argv);
    $microtime_start = microtime(true);
    $r = call_user_func_array($fctName,$argv);
    $diff_microsec = microtime(true)-$microtime_start - $diff_microsec_dummy;
    if($diff_microsec < 0.0) $diff_microsec = 0.0;
    if($diff_microsec >= 1.0) $strDiff = "[+". sprintf('%1.3F',$diff_microsec)." s]";
    elseif($diff_microsec >= 0.010) $strDiff = "[+".(int)($diff_microsec*1000)." ms]";
    else $strDiff = "[+".(int)($diff_microsec*1000000)." µs]";
    
    if(! is_string($fctName)) $fctName = self::TypeInfo($fctName);
    self::write('call_user_func: '.$fctName, $argv, 'return:',$r,$strDiff);
    return $r;
  }

  /*
   * check Methode
   */
  public static function classfkt($classInstance,$fctName/** $var1, $var2, .. **/){
    $argv = func_get_args();
    array_shift($argv);
    array_shift($argv);
    $callBackArr = array($classInstance,$fctName);
    
    $microtime_start = microtime(true);
    $r = call_user_func_array($callBackArr,$argv);
    $microtime_end = microtime(true);
    $diff_microsec = $microtime_end-$microtime_start;
    if($diff_microsec >= 1.0) $strDiff = "[+". sprintf('%1.3F',$diff_microsec)." s]";
    elseif($diff_microsec >= 0.010) $strDiff = "[+".(int)($diff_microsec*1000)." ms]";
    else $strDiff = "[+".(int)($diff_microsec*1000000)." µs]";
    self::write(get_class($classInstance).'->'.$fctName.'(', $argv, 'return:',$r,'Time: '.$strDiff);
    return $r;
  }
  
  public static function setTitleColor($backgroundColor) {
    self::$trHeadStyle = preg_replace('/background-color:.+;/','background:'.$backgroundColor.';',self::$trHeadStyle);
  }

 /*
  * non public functions
  */

 /*
  * function returns a string with non-printable characters
  * are shown in hexadecimal notation \xzz (z = number)
  * Do not use external
  */ 
  protected static function str2print($s) {
    $s = substr($s,0,self::$stringCut);
    if(preg_match("//u",$s)) {
      $regEx = '/[\p{C}]/usS';
    }
    else {
      $regEx = '/[\x0-\x1f\x7f-\xff]/sS';
    }
    $s = preg_replace_callback(
      $regEx,
      function($m) {
        $hx = '\\x'.implode('\\x',str_split(bin2hex($m[0]),2));
        return '{~+~}'.$hx.'{~-~}';
      }, 
      $s
    );
    return $s;
  }

  protected static function displayAndLog($argv,$backtrace)
  {
    self::$recbuf .= self::recArg($argv,$backtrace);  //save current info
    if(self::$logfilename == "") 
    { //empty logfilename -> display
      //if (!headers_sent()) header('Content-Type: text/html; charset=UTF-8');
      echo self::$recbuf;
      if(self::$real_time_output) {
         echo (str_repeat(' ',4096))."\r";
      }
      flush();
    }
    else 
    { //write logfile
      file_put_contents(self::$logfilename,self::$recbuf,FILE_APPEND);
    }
    self::$recbuf = "";
  }

  protected static function backtraceInfo($backtrace)
  {
    $backtraceKeys = array('class','object','type','function');  //'file','args','type'
    $cutAfterChars = ':(';  //strings from arguments and objects cut after this chars 
    $info = "";
    foreach($backtrace as $i => $bi) {
      if($i == 0) 
      {
        $fromFile = isset($bi['file']) ? $bi['file'] : '';
        $info .= " ".$bi['class'].$bi['type'].$bi['function'];
        $info .= ' "'.basename($fromFile).'" Line '.$bi['line'];
      }
      else
      {
        $fromFileNew = isset($bi['file']) ? $bi['file'] : '';
        if($fromFileNew != '' && $fromFileNew != $fromFile) {
          $fromFile = $fromFileNew;
          $fromFileInfo = '"'.basename($fromFile).'"';
        }
        else {
          $fromFileInfo = '';
        }
        if(array_key_exists('line',$bi)) $info .= " <=".$fromFileInfo." Line ".$bi["line"];
        //$info .= " &lt;= Line ".$bi["line"];
        foreach($backtraceKeys as $k) 
        {
          if(isset($bi[$k])) 
          {
            $info .= " ".(is_string($bi[$k]) ? $bi[$k] : '{'.preg_replace('/['.$cutAfterChars.'].*/s','',var_export($bi[$k],true)).'}');
            if($k == 'function') 
            {
              $args = "(";
              if(isset($bi['args']))
              {
                foreach($bi['args'] as $arg) {
                  if(is_scalar($arg)) {  //01.08.2014
                    $args .= ($bi[$k] === 'include') 
                      ? var_export($arg,true) 
                      :preg_replace('/['.$cutAfterChars.'].*/s','',var_export($arg,true)).", "
                    ;
                  } 
                  else {
                    $args .= gettype($arg).',';
                  }
                }
                $info .= rtrim($args,", ").")";
              }
            }
          }
        }
      }
    } 
    return $info;
  }
  
  protected static function recArg($argv, $backtrace) {
    //$btel = array('class','object','type','function');  //'file','args','type'

    $recadd = '<table style="'.self::$tableStyle.'">'."\r\n".
      '<tr style="'.self::$trHeadStyle.'">'."\r\n".
      '<td colspan="3"><b>';
    $microtime_float = microtime(true);
    //times
    $recadd .= " [".date("d.m.Y H:i:s",(int)$microtime_float).",".sprintf("%03d", (int)(fmod($microtime_float,1) * 1000))."]";
    if(self::$lastTime > 0) {
      $diff_microsec = $microtime_float-self::$lastTime;
      if($diff_microsec >= 1.0) $recadd .= "[+". sprintf('%1.3F',$diff_microsec)." s]";
      elseif($diff_microsec >= 0.010) $recadd .= "[+".(int)($diff_microsec*1000)." ms]";
      else $recadd .= "[+".(int)($diff_microsec*1000000)." &mu;s]";
    }
    self::$lastTime = $microtime_float;
    //mem
    $recadd .= '('.(int)(memory_get_usage(false)/1024). 'k/'.(int)(memory_get_peak_usage(false)/1024).'k)'; //kByte
    //backtraceInfo
    $recadd .= self::esc(self::backtraceInfo($backtrace));
    $recadd .= "</b></td></tr>"."\r\n";
    //vars
    foreach($argv as $k => $arg){
      $recadd .= '<tr><td style="width:'.self::$col1width.';'.self::$tdStyle.'"><b> '.
        $k.'</b></td><td style="width:'.self::$col2width.';'.self::$tdStyle.'">';
      $pre = is_object($arg) || is_array($arg) ? '<pre style="display:inline">' : "";
      $recadd .= self::TypeInfo($arg).'</td><td style="'.self::$tdStyle.'">'.$pre;
      if(is_int($arg)) {
        $t = $arg." [".strtoupper(sprintf("%08x", $arg))."h]";
      }
      elseif(is_string($arg)) {
        $t = self::$showSpecialChars ? ('"'.self::str2print($arg).'"') : $arg;
      }
      else {
        if($arg instanceof SimpleXMLElement) {
          $t = $arg->asXML();
          if(!preg_match("/<.+>/",$t)) {
            $t = var_export($arg,true);
            $r = preg_match("/state\((.*)\)/s",$t,$match);
            if($r) $t = $match[1];
          }
        }
        elseif ($arg instanceof DOMElement && true) {
          //neu 19.6.2013
          //$newdoc = new DOMDocument();
          //$node = $newdoc->importNode($arg, true);
          //$newdoc->appendChild($node);
          //$t = $newdoc->saveXML($node);
          $t = $arg->ownerDocument->saveXML($arg);
        }
        elseif ($arg instanceof DOMDocument && true) {
          //neu 19.6.2013
          $arg->formatOutput = true;
          $t = $arg->saveXML();
        }

        else $t = var_export($arg,true);
        $t = str_ireplace('stdClass::__set_state','(object)',$t);  //ab V1.8
        $t = str_replace("' . \"\\0\" . '",chr(0),$t);  //Exot ."\0". entfernen
        //Filter Output
        if(self::$showSpecialChars) $t = preg_replace_callback(
          "/=> '([^']+)'/",
          'self::cb1',  //'self::cb1'
          $t
          );
      }
      $recadd .= str_replace(
        array('{~+~}','{~-~}'),
        array('<span style="background:#ccc;margin:1px;">','</span>'),
        self::esc($t)
        );
      $recadd .= ($pre ? '</pre>' : '')."</td></tr>";
    }
    $recadd .= "</table>"."\r\n";
    return $recadd;
  }
  
  protected static function cb1($m){
    return '=> "'.self::str2print($m[1]).'"';
  }
  
  protected static function closeLog() {
    if(self::$logfilename != "") {
      file_put_contents(self::$logfilename,str_replace("\n[","<br>[",file_get_contents(self::$logfilename)));
      if( !self::$log_file_append) file_put_contents(self::$logfilename,'</body></html>',FILE_APPEND);
    }
    self::$logfilename = '';
  }
  
  //Escape
  protected static function esc($s){
    return htmlspecialchars($s, ENT_QUOTES |ENT_IGNORE, 'UTF-8');
  }
  

}
